<?php
defined('_JEXEC') or die;

/**
 * Circle Images (3D) v1.1.4
 *
 * Fix for "only 3 visible when rotating":
 * - Do NOT rotate the ring with CSS transform animation (that rotates the faces away from the camera).
 * - Use JS to update each item's angle and counter-rotate the face so every image keeps facing front.
 * - Also set z-index/scale/opacity based on depth for "optimal view".
 */

$props = $props ?? [];
$attrs = $attrs ?? [];

$preset = (string) ($props['preset'] ?? 'standard');
$preset = ($preset === 'custom') ? 'custom' : 'standard';

$countProp = $props['count'] ?? 'auto';
$count = 0;
if (is_string($countProp) && strtolower(trim($countProp)) === 'auto') {
  $count = 0; // decide later based on images
} else {
  $count = (int) $countProp;
}
$count = max(0, min(36, $count));
$autoOptimize = !empty($props['auto_optimize']);
$fillEmpty = !empty($props['fill_empty']);

$rotate = !empty($props['rotate']);
$rotateDir = (($props['rotate_dir'] ?? 'cw') === 'ccw') ? 'ccw' : 'cw';
$rotateSpeed = max(2, (int) ($props['rotate_speed'] ?? 18)); // seconds per 360°

// Margins
$mt  = (int) ($props['mt'] ?? 0);
$mb  = (int) ($props['mb'] ?? 0);
$mtS = (int) ($props['mt_s'] ?? 0);
$mbS = (int) ($props['mb_s'] ?? 0);


// Settings (preset)
if ($preset === 'standard') {

  $radiusBase = 260;
  $gap = 0;
  $radiusBaseS = 0;
  $gapS = 0;

  $imgW = 140;
  $imgH = 140;
  $imgWs = 0;
  $imgHs = 0;

  $persp = 1200;
  $tilt = 0;

  $shape = 'circle';

} else {

  $radiusBase = (int) ($props['radius'] ?? 260);
  $gap = (int) ($props['gap'] ?? 0);
  $radiusBaseS = (int) ($props['radius_s'] ?? 0);
  $gapS = (int) ($props['gap_s'] ?? 0);

  $imgW = (int) ($props['img_w'] ?? 140);
  $imgH = (int) ($props['img_h'] ?? 140);
  $imgWs = (int) ($props['img_w_s'] ?? 0);
  $imgHs = (int) ($props['img_h_s'] ?? 0);

  $persp = (int) ($props['perspective'] ?? 1200);
  $tilt = (int) ($props['tilt'] ?? 0);

  $shape = (string) ($props['round'] ?? 'circle');
}

$radiusBase = max(0, $radiusBase);
$gap = max(0, $gap);
$radiusBaseS = max(0, (int) $radiusBaseS);
$gapS = max(0, (int) $gapS);

$imgW = max(10, $imgW);
$imgH = max(10, $imgH);
$persp = max(200, $persp);

// border radius
$br = '0';
if ($shape === 'circle') $br = '999px';
elseif ($shape === 'rounded') $br = '16px';

// Robust src extraction
$pickSrc = function ($img): string {

  if (is_array($img)) {
    foreach (['src','url','path','image','value'] as $k) {
      if (!empty($img[$k]) && is_string($img[$k])) return trim((string) $img[$k]);
    }
    return '';
  }

  if (is_object($img)) {
    foreach (['src','url','path','image','value'] as $k) {
      if (isset($img->$k) && is_string($img->$k) && trim((string) $img->$k) !== '') return trim((string) $img->$k);
    }
    if (method_exists($img, '__toString')) return trim((string) $img);
    return '';
  }

  return is_string($img) ? trim($img) : '';
};

// Images: prefer content-items children (media import), fallback to legacy Image 1..6.
$childImages = [];
if (!empty($children) && is_array($children)) {
  foreach ($children as $child) {
    $cp = (array) ($child->props ?? []);
    $src = $pickSrc($cp['image'] ?? '');
    if ($src !== '') {
      $childImages[] = $src;
    }
  }
}

// Use at most 36 images
$childImages = array_slice($childImages, 0, 36);

$legacyImages = [
  $pickSrc($props['image1'] ?? ''),
  $pickSrc($props['image2'] ?? ''),
  $pickSrc($props['image3'] ?? ''),
  $pickSrc($props['image4'] ?? ''),
  $pickSrc($props['image5'] ?? ''),
  $pickSrc($props['image6'] ?? ''),
];

// Prefer children if present
$images = array_values(array_filter($childImages, fn($s) => $s !== ''));
if (!$images) {
  $images = array_values(array_filter($legacyImages, fn($s) => $s !== ''));
}
// Resolve auto count after images are known
$imagesLen = is_array($images) ? count($images) : 0;
if ($count === 0) {
  // Auto: use number of uploaded images (max 36). If none and placeholders enabled, default to 6.
  $count = $imagesLen > 0 ? min(36, $imagesLen) : (!empty($fillEmpty) ? 6 : 0);
}
$count = max(0, min(36, (int) $count));

// If placeholders are disabled, don't render empty positions
if (empty($fillEmpty) && $imagesLen > 0) {
  $count = min($count, $imagesLen);
}
$count = max(1, $count);

// Effective radius (+gap) and auto optimize
$radiusWanted = ($count === 1) ? 0 : ($radiusBase + $gap);
$radiusEff = $radiusWanted;

if ($autoOptimize && $count > 1) {

  $maxDim = max($imgW, $imgH);
  $need = $maxDim + $gap;
  $sin = sin(M_PI / $count);

  if ($sin > 0) {
    $rMin = (int) ceil($need / (2 * $sin));
    $radiusEff = max($radiusWanted, $rMin);
  }
}

// Mobile size
$imgWm = $imgWs > 0 ? max(10, $imgWs) : max(10, (int) round($imgW * 0.78));
$imgHm = $imgHs > 0 ? max(10, $imgHs) : max(10, (int) round($imgH * 0.78));

$radiusBaseM = ($radiusBaseS > 0) ? $radiusBaseS : (int) round($radiusBase * 0.72);
$gapM = ($gapS > 0) ? $gapS : $gap;
$radiusWantedM = ($count === 1) ? 0 : ($radiusBaseM + $gapM);
$radiusEffM = $radiusWantedM;

if ($autoOptimize && $count > 1) {
  $maxDimM = max($imgWm, $imgHm);
  $needM = $maxDimM + $gapM;
  $sin = sin(M_PI / $count);
  if ($sin > 0) {
    $rMinM = (int) ceil($needM / (2 * $sin));
    $radiusEffM = max($radiusWantedM, $rMinM);
  }
}

// Box sizes
$box  = (2 * $radiusEff)  + max($imgW,  $imgH)  + 40;
$boxm = (2 * $radiusEffM) + max($imgWm, $imgHm) + 40;

// UID for scoping
$uid = 'hpbci3d-' . substr(md5(json_encode($props) . uniqid('', true)), 0, 12);
$uid = preg_replace('~[^a-zA-Z0-9\-_]~', '', $uid);

// Wrapper
$wrap = $this->el('div', [
  'class' => [
    'hpb-circle-3d',
    $uid,
  ],
]);

?>
<?= $wrap($props, $attrs) ?>

<style>
.<?= $uid ?>{
  --hpb3d-radius: <?= (int) $radiusEff ?>px;
  --hpb3d-w: <?= (int) $imgW ?>px;
  --hpb3d-h: <?= (int) $imgH ?>px;
  --hpb3d-persp: <?= (int) $persp ?>px;
  --hpb3d-tilt: <?= (int) $tilt ?>deg;
  --hpb3d-br: <?= $br ?>;
  --hpb3d-box: <?= (int) $box ?>px;
  --hpb3d-mt: <?= (int) $mt ?>px;
  --hpb3d-mb: <?= (int) $mb ?>px;
}

.<?= $uid ?> .hpb3d-stage{
  width: 100%;
  max-width: var(--hpb3d-box);
  height: var(--hpb3d-box);
  margin-left: auto;
  margin-right: auto;
  margin-top: var(--hpb3d-mt);
  margin-bottom: var(--hpb3d-mb);
  display: flex;
  align-items: center;
  justify-content: center;
  perspective: var(--hpb3d-persp);
  perspective-origin: 50% 50%;
  transform-style: preserve-3d;
  contain: layout paint size;
  transform: translateZ(0);
}

.<?= $uid ?> [data-pause-tap="1"]{ cursor: pointer; }

.<?= $uid ?> .hpb3d-ring{
  position: relative;
  width: 0;
  height: 0;
  transform-style: preserve-3d;
  transform: rotateX(var(--hpb3d-tilt)) translateZ(0);
  will-change: transform;
}

.<?= $uid ?> .hpb3d-item{
  position: absolute;
  left: 50%;
  top: 50%;
  width: var(--hpb3d-w);
  height: var(--hpb3d-h);
  transform-style: preserve-3d;
  transform: translate3d(-50%, -50%, 0);
  transform-origin: 50% 50%;
  backface-visibility: hidden;
  will-change: transform, opacity;
}

.<?= $uid ?> .hpb3d-face{
  width: 100%;
  height: 100%;
  transform-style: preserve-3d;
  transform-origin: 50% 50%;
  backface-visibility: hidden;
  will-change: transform;
}

.<?= $uid ?> .hpb3d-img{
  width: 100%;
  height: 100%;
  display: block;
  object-fit: cover;
  border-radius: var(--hpb3d-br);
  box-shadow: 0 12px 35px rgba(0,0,0,.22);
  background: rgba(0,0,0,.05);
  transform: translateZ(0.01px);
}

.<?= $uid ?> .hpb3d-ph{
  width: 100%;
  height: 100%;
  border-radius: var(--hpb3d-br);
  background: rgba(0,0,0,.06);
  box-shadow: 0 12px 35px rgba(0,0,0,.12);
  display: flex;
  align-items: center;
  justify-content: center;
  font-size: 12px;
  opacity: .6;
}

@media (max-width: 640px){
  .<?= $uid ?>{
    --hpb3d-radius: <?= (int) $radiusEffM ?>px;
    --hpb3d-w: <?= (int) $imgWm ?>px;
    --hpb3d-h: <?= (int) $imgHm ?>px;
    --hpb3d-box: <?= (int) $boxm ?>px;
    --hpb3d-mt: <?= (int) $mtS ?>px;
    --hpb3d-mb: <?= (int) $mbS ?>px;
  }
}
</style>

<div class="hpb3d-stage" data-hpb3d="1"
  data-count="<?= (int) $count ?>"
  data-rotate="<?= $rotate ? '1' : '0' ?>"
  data-dir="<?= htmlspecialchars($rotateDir, ENT_QUOTES) ?>"
  data-speed="<?= (int) $rotateSpeed ?>"
  data-pause-hover="<?= !empty($props['pause_hover']) ? '1' : '0' ?>"
  data-pause-tap="<?= !empty($props['pause_tap']) ? '1' : '0' ?>"
>
  <div class="hpb3d-ring">
    <?php for ($i = 0; $i < $count; $i++) :
      $src = trim((string) ($images[$i] ?? ''));
      $label = 'Image ' . ($i + 1);
    ?>
      <div class="hpb3d-item" data-base="<?= (float) (360 / $count * $i) ?>">
        <div class="hpb3d-face">
          <?php if ($src !== '') : ?>
            <img class="hpb3d-img" src="<?= htmlspecialchars($src, ENT_QUOTES) ?>" alt="<?= htmlspecialchars($label, ENT_QUOTES) ?>" loading="lazy">
          <?php elseif ($fillEmpty) : ?>
            <div class="hpb3d-ph"><?= htmlspecialchars($label, ENT_QUOTES) ?></div>
          <?php endif; ?>
        </div>
      </div>
    <?php endfor; ?>
  </div>
</div>

<script>
(function(){
  const root = document.currentScript && document.currentScript.previousElementSibling
    ? document.currentScript.previousElementSibling
    : null;

  // Fallback: find nearest data-hpb3d in parent
  const host = root && root.matches && root.matches('[data-hpb3d]') ? root : (document.currentScript ? document.currentScript.parentElement.querySelector('[data-hpb3d]') : null);
  if (!host) return;

  if (host.dataset.hpb3dInit) return;
  host.dataset.hpb3dInit = "1";

  const ring = host.querySelector('.hpb3d-ring');
  const items = Array.from(host.querySelectorAll('.hpb3d-item'));
  const n = items.length;

  const rotate = host.dataset.rotate === "1";
  const dir = host.dataset.dir === "ccw" ? "ccw" : "cw";
  const speed = Math.max(2, parseInt(host.dataset.speed || "18", 10)); // seconds per 360
  const pauseHover = host.dataset.pauseHover === "1";
  const pauseTap = host.dataset.pauseTap === "1";
  let paused = false;

  const prefersReduce = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;

  function getVarPx(el, name) {
    const v = getComputedStyle(el).getPropertyValue(name).trim();
    const m = v.match(/^(-?\d+(\.\d+)?)px$/);
    return m ? parseFloat(m[1]) : 0;
  }

  // Cache CSS vars to avoid per-frame wobble from repeated computed-style reads
  const state = { radius: 0 };
  function syncVars(){
    state.radius = getVarPx(host, '--hpb3d-radius');
    if (!state.radius || state.radius < 0) state.radius = 0;
  }
  syncVars();
  window.addEventListener('resize', () => { syncVars(); }, { passive: true });

  // Precompute base angles once
  const bases = items.map((it, idx) => {
    const b = parseFloat(it.getAttribute('data-base') || (idx * (360/n)));
    return isFinite(b) ? b : (idx * (360/n));
  });

  function setPose(offsetDeg) {
    const radius = state.radius;

    for (let idx = 0; idx < items.length; idx++) {
      const it = items[idx];
      let a = bases[idx] + offsetDeg;

      // normalize to [0..360)
      a = ((a % 360) + 360) % 360;

      const rad = a * Math.PI / 180;
      const depth = Math.cos(rad); // 1 front, -1 back

      // No size pumping (prevents "wobble"). Only depth via opacity + z-index.
      const opacity = 0.50 + ((depth + 1) / 2) * 0.50; // 0.50..1.0
      const zIndex = Math.round((depth + 1) * 1000);

      const aStr = a.toFixed(3);

      // Use translate3d to keep GPU stable
      it.style.transform =
        `translate3d(-50%, -50%, 0) rotateY(${aStr}deg) translateZ(${radius}px)`;

      const face = it.querySelector('.hpb3d-face');
      if (face) face.style.transform = `rotateY(${-a}deg)`;

      it.style.opacity = opacity.toFixed(3);
      it.style.zIndex = zIndex.toString();
    }
  }
  // Initial pose
  setPose(0);
  host.setAttribute('aria-busy', 'false');

  if (!rotate || prefersReduce || n < 2) return;

  // Pause on hover (desktop)
  if (pauseHover) {
    host.addEventListener('mouseenter', () => { paused = true; }, { passive: true });
    host.addEventListener('mouseleave', () => { paused = false; }, { passive: true });
  }

  // Toggle pause on tap/click (mobile)
  if (pauseTap) {
    host.addEventListener('click', (e) => {
      // Ignore if user selects text etc.
      paused = !paused;
      host.setAttribute('aria-busy', paused ? 'true' : 'false');
    });
  }

  let offset = 0;
  let last = performance.now();
  const sign = (dir === "ccw") ? 1 : -1;

  function tick(now) {
    const dt = Math.min(0.05, (now - last) / 1000);
    last = now;

    if (!paused) {
      // deg per second: 360 / speed
      offset += sign * (360 / speed) * dt;
    }
    // normalize
    if (offset > 360 || offset < -360) offset = offset % 360;

    setPose(offset);
    requestAnimationFrame(tick);
  }

  requestAnimationFrame(tick);
})();
</script>

<?= $wrap->end() ?>
