<?php
defined('_JEXEC') or die;

/**
 * Circle Images (3D)
 * Adds: stop-at-image (stop rotation when selected image is in front).
 *
 * Notes:
 * - "Stop at image" matches by the raw image src/path stored by YOOtheme.
 * - If not found, "Stop at index (fallback)" is used (1-based index).
 */

$props = $props ?? [];
$attrs = $attrs ?? [];

/** @var array<int,object> $children */
$children = isset($children) && is_array($children) ? $children : [];

$preset = (($props['preset'] ?? 'standard') === 'custom') ? 'custom' : 'standard';

$countProp = $props['count'] ?? 'auto';
if (is_string($countProp) && strtolower(trim($countProp)) === 'auto') {
    $count = 0; // decide later
} else {
    $count = (int) $countProp;
}
$count = max(0, min(36, $count));

$autoOptimize = !empty($props['auto_optimize']);
$fillEmpty = !empty($props['fill_empty']);

$rotate = !empty($props['rotate']);
$rotateDir = (($props['rotate_dir'] ?? 'cw') === 'ccw') ? 'ccw' : 'cw';
$rotateSpeed = max(2, (int) ($props['rotate_speed'] ?? 18)); // seconds per 360°

$pauseHover = !empty($props['pause_hover']);
$pauseTap   = !empty($props['pause_tap']);

// Margins
$mt  = (int) ($props['mt'] ?? 0);
$mb  = (int) ($props['mb'] ?? 0);
$mtS = (int) ($props['mt_s'] ?? 0);
$mbS = (int) ($props['mb_s'] ?? 0);

// Robust src extraction
$pickSrc = static function ($img): string {

    if (is_array($img)) {
        foreach (['src','url','path','image','value'] as $k) {
            if (!empty($img[$k]) && is_string($img[$k])) return trim((string) $img[$k]);
        }
        // some nested formats: [{src:...}]
        if (!empty($img[0]) && is_array($img[0])) {
            foreach (['src','url','path'] as $k) {
                if (!empty($img[0][$k]) && is_string($img[0][$k])) return trim((string) $img[0][$k]);
            }
        }
        return '';
    }

    if (is_object($img)) {
        foreach (['src','url','path','image','value'] as $k) {
            if (isset($img->$k) && is_string($img->$k) && trim((string) $img->$k) !== '') return trim((string) $img->$k);
        }
        if (method_exists($img, '__toString')) return trim((string) $img);
        return '';
    }

    return is_string($img) ? trim($img) : '';
};

// Settings (preset)
if ($preset === 'standard') {

    $radiusBase  = 260;
    $gap         = 0;
    $radiusBaseS = 0;
    $gapS        = 0;

    $imgW  = 140;
    $imgH  = 140;
    $imgWs = 0;
    $imgHs = 0;

    $persp = 1200;
    $tilt  = 0;

    $shape = 'circle';

} else {

    $radiusBase  = (int) ($props['radius'] ?? 260);
    $gap         = (int) ($props['gap'] ?? 0);
    $radiusBaseS = (int) ($props['radius_s'] ?? 0);
    $gapS        = (int) ($props['gap_s'] ?? 0);

    $imgW  = (int) ($props['img_w'] ?? 140);
    $imgH  = (int) ($props['img_h'] ?? 140);
    $imgWs = (int) ($props['img_w_s'] ?? 0);
    $imgHs = (int) ($props['img_h_s'] ?? 0);

    $persp = (int) ($props['perspective'] ?? 1200);
    $tilt  = (int) ($props['tilt'] ?? 0);

    $shape = (string) ($props['round'] ?? 'circle');
}

$radiusBase  = max(0, $radiusBase);
$gap         = max(0, $gap);
$radiusBaseS = max(0, (int) $radiusBaseS);
$gapS        = max(0, (int) $gapS);

$imgW  = max(10, (int) $imgW);
$imgH  = max(10, (int) $imgH);
$persp = max(200, (int) $persp);

// Border radius
$br = '0';
if ($shape === 'circle') $br = '999px';
elseif ($shape === 'rounded') $br = '16px';

// Images: prefer content-items children (media import), fallback to legacy Image 1..6.
$childImages = [];
foreach ($children as $child) {
    $cp  = (array) ($child->props ?? []);
    $src = $pickSrc($cp['image'] ?? '');
    if ($src !== '') {
        $childImages[] = $src;
    }
}
$childImages = array_slice($childImages, 0, 36);

$legacyImages = [
    $pickSrc($props['image1'] ?? ''),
    $pickSrc($props['image2'] ?? ''),
    $pickSrc($props['image3'] ?? ''),
    $pickSrc($props['image4'] ?? ''),
    $pickSrc($props['image5'] ?? ''),
    $pickSrc($props['image6'] ?? ''),
];

$images = array_values(array_filter($childImages, static fn($s) => $s !== ''));
if (!$images) {
    $images = array_values(array_filter($legacyImages, static fn($s) => $s !== ''));
}

// Resolve auto count after images are known
$imagesLen = count($images);
if ($count === 0) {
    $count = $imagesLen > 0 ? min(36, $imagesLen) : (!empty($fillEmpty) ? 6 : 0);
}
$count = max(0, min(36, (int) $count));

// If placeholders are disabled, don't render empty positions
if (empty($fillEmpty) && $imagesLen > 0) {
    $count = min($count, $imagesLen);
}
$count = max(1, $count);

// Effective radius (+gap) and auto optimize
$radiusWanted = ($count === 1) ? 0 : ($radiusBase + $gap);
$radiusEff = $radiusWanted;

if ($autoOptimize && $count > 1) {
    $maxDim = max($imgW, $imgH);
    $need   = $maxDim + $gap;
    $sin    = sin(M_PI / $count);
    if ($sin > 0) {
        $rMin = (int) ceil($need / (2 * $sin));
        $radiusEff = max($radiusWanted, $rMin);
    }
}

// Mobile size
$imgWm = $imgWs > 0 ? max(10, $imgWs) : max(10, (int) round($imgW * 0.78));
$imgHm = $imgHs > 0 ? max(10, $imgHs) : max(10, (int) round($imgH * 0.78));

$radiusBaseM = ($radiusBaseS > 0) ? $radiusBaseS : (int) round($radiusBase * 0.72);
$gapM        = ($gapS > 0) ? $gapS : $gap;

$radiusWantedM = ($count === 1) ? 0 : ($radiusBaseM + $gapM);
$radiusEffM = $radiusWantedM;

if ($autoOptimize && $count > 1) {
    $maxDimM = max($imgWm, $imgHm);
    $needM   = $maxDimM + $gapM;
    $sin     = sin(M_PI / $count);
    if ($sin > 0) {
        $rMinM = (int) ceil($needM / (2 * $sin));
        $radiusEffM = max($radiusWantedM, $rMinM);
    }
}

// Box sizes
$box  = (2 * $radiusEff)  + max($imgW,  $imgH)  + 40;
$boxm = (2 * $radiusEffM) + max($imgWm, $imgHm) + 40;

// Build slots (count positions)
$slots = [];
for ($i = 0; $i < $count; $i++) {
    $src = $images[$i] ?? '';
    $slots[] = [
        'src' => $src,
        'is_placeholder' => ($src === ''),
        'index' => $i + 1,
    ];
}

// Stop-at config
$stopImageSrc = $pickSrc($props['stop_image'] ?? '');
$stopIndexFallback = (int) ($props['stop_index'] ?? 0);
$stopIndexFallback = max(0, min(36, $stopIndexFallback));

$stopIndex = 0;
if ($stopImageSrc !== '') {
    foreach ($slots as $slot) {
        if (($slot['src'] ?? '') === $stopImageSrc) {
            $stopIndex = (int) $slot['index'];
            break;
        }
    }
}
if ($stopIndex === 0 && $stopIndexFallback > 0) {
    $stopIndex = min($count, $stopIndexFallback);
}

// Stop pause duration (seconds)
$stopPauseSec = (float) ($props['stop_pause'] ?? 0);
$stopPauseSec = max(0.0, min(600.0, $stopPauseSec));

// Zoom pulse when stopped (auto-stop)
$stopZoom = !empty($props['stop_zoom']);
$stopZoomMin = (float) ($props['stop_zoom_min'] ?? 0.92);
$stopZoomMax = (float) ($props['stop_zoom_max'] ?? 1.08);
$stopZoomCycle = (float) ($props['stop_zoom_cycle'] ?? 1.2);

$stopZoomMin = max(0.5, min(1.0, $stopZoomMin));
$stopZoomMax = max(1.0, min(1.6, $stopZoomMax));
if ($stopZoomMin >= $stopZoomMax) {
    // Ensure a usable gap
    $stopZoomMin = max(0.5, min(0.95, $stopZoomMax - 0.05));
}
$stopZoomCycle = max(0.2, min(20.0, $stopZoomCycle));


// UID for scoping
$uid = 'hpbci3d-' . substr(md5(json_encode($props) . uniqid('', true)), 0, 12);
$uid = preg_replace('~[^a-zA-Z0-9\-_]~', '', $uid);

// Wrapper
$wrap = $this->el('div', [
    'class' => [
        'hpb-circle-3d',
        $uid,
    ],
]);

echo $wrap($props, $attrs);

?>

<style>
.<?= $uid ?>{
  --hpb3d-radius: <?= (int) $radiusEff ?>px;
  --hpb3d-w: <?= (int) $imgW ?>px;
  --hpb3d-h: <?= (int) $imgH ?>px;
  --hpb3d-persp: <?= (int) $persp ?>px;
  --hpb3d-tilt: <?= (int) $tilt ?>deg;
  --hpb3d-br: <?= $br ?>;
  --hpb3d-box: <?= (int) $box ?>px;
  --hpb3d-mt: <?= (int) $mt ?>px;
  --hpb3d-mb: <?= (int) $mb ?>px;
  --hpbci-stop-zoom-min: <?= htmlspecialchars(number_format($stopZoomMin, 3, '.', ''), ENT_QUOTES) ?>;
  --hpbci-stop-zoom-max: <?= htmlspecialchars(number_format($stopZoomMax, 3, '.', ''), ENT_QUOTES) ?>;
  --hpbci-stop-zoom-cycle: <?= htmlspecialchars(number_format($stopZoomCycle, 3, '.', ''), ENT_QUOTES) ?>s;
}

@media (max-width: 640px){
  .<?= $uid ?>{
    --hpb3d-radius: <?= (int) $radiusEffM ?>px;
    --hpb3d-w: <?= (int) $imgWm ?>px;
    --hpb3d-h: <?= (int) $imgHm ?>px;
    --hpb3d-box: <?= (int) $boxm ?>px;
    --hpb3d-mt: <?= (int) $mtS ?>px;
    --hpb3d-mb: <?= (int) $mbS ?>px;
  }
}

.<?= $uid ?> .hpb3d-stage{
  width: 100%;
  max-width: var(--hpb3d-box);
  height: var(--hpb3d-box);
  margin-left: auto;
  margin-right: auto;
  margin-top: var(--hpb3d-mt);
  margin-bottom: var(--hpb3d-mb);
  display: flex;
  align-items: center;
  justify-content: center;
  perspective: var(--hpb3d-persp);
  perspective-origin: 50% 50%;
  transform-style: preserve-3d;
  contain: layout paint size;
  transform: translateZ(0);
  user-select: none;
}

.<?= $uid ?> [data-pause-tap="1"]{ cursor: pointer; }

.<?= $uid ?> .hpb3d-ring{
  position: relative;
  width: 0;
  height: 0;
  transform-style: preserve-3d;
  transform: rotateX(var(--hpb3d-tilt)) translateZ(0);
  will-change: transform;
}

.<?= $uid ?> .hpb3d-item{
  position: absolute;
  left: 50%;
  top: 50%;
  width: var(--hpb3d-w);
  height: var(--hpb3d-h);
  transform-style: preserve-3d;
  transform: translate3d(-50%, -50%, 0);
  transform-origin: 50% 50%;
  backface-visibility: hidden;
  will-change: transform, opacity;
}

.<?= $uid ?> .hpb3d-card{
  width: 100%;
  height: 100%;
  border-radius: var(--hpb3d-br);
  overflow: hidden;
  background: rgba(0,0,0,.06);
  box-shadow: 0 14px 35px rgba(0,0,0,.16);
  transform: translateZ(0) scale(1);
}

.<?= $uid ?> .hpb3d-card img{
  width: 100%;
  height: 100%;
  object-fit: cover;
  display: block;
}


.<?= $uid ?> .hpb3d-card.hpbci-stop-zoom{
  animation: hpbci-stop-zoom-pulse var(--hpbci-stop-zoom-cycle, 1.2s) ease-in-out infinite;
  will-change: transform;
}

@media (prefers-reduced-motion: reduce){
  .<?= $uid ?> .hpb3d-card.hpbci-stop-zoom{ animation: none; }
}

@keyframes hpbci-stop-zoom-pulse{
  0%{   transform: translateZ(0) scale(1); }
  25%{  transform: translateZ(0) scale(var(--hpbci-stop-zoom-min, 0.92)); }
  50%{  transform: translateZ(0) scale(var(--hpbci-stop-zoom-max, 1.08)); }
  75%{  transform: translateZ(0) scale(var(--hpbci-stop-zoom-min, 0.92)); }
  100%{ transform: translateZ(0) scale(1); }
}

.<?= $uid ?> .hpb3d-placeholder{
  width: 100%;
  height: 100%;
  display: grid;
  place-items: center;
  font-size: 14px;
  letter-spacing: .08em;
  text-transform: uppercase;
  color: rgba(0,0,0,.55);
  background:
    radial-gradient(circle at 35% 30%, rgba(255,255,255,.8), rgba(255,255,255,0) 60%),
    linear-gradient(135deg, rgba(0,0,0,.08), rgba(0,0,0,.02));
}
</style>

<div class="hpb3d-stage"
     data-rotate="<?= $rotate ? '1' : '0' ?>"
     data-dir="<?= htmlspecialchars($rotateDir, ENT_QUOTES) ?>"
     data-speed="<?= (int) $rotateSpeed ?>"
     data-pause-hover="<?= $pauseHover ? '1' : '0' ?>"
     data-pause-tap="<?= $pauseTap ? '1' : '0' ?>"
     data-stop-index="<?= (int) $stopIndex ?>"
     data-stop-pause="<?= htmlspecialchars((string) $stopPauseSec, ENT_QUOTES) ?>"
     data-stop-zoom="<?= $stopZoom ? '1' : '0' ?>"
>
  <div class="hpb3d-ring">
    <?php foreach ($slots as $slot): ?>
      <div class="hpb3d-item" data-i="<?= (int) $slot['index'] ?>">
        <div class="hpb3d-card">
          <?php if (!$slot['is_placeholder']): ?>
            <img src="<?= htmlspecialchars((string) $slot['src'], ENT_QUOTES) ?>" alt="" loading="lazy" decoding="async">
          <?php else: ?>
            <div class="hpb3d-placeholder">Image <?= (int) $slot['index'] ?></div>
          <?php endif; ?>
        </div>
      </div>
    <?php endforeach; ?>
  </div>
</div>

<script>
(() => {
  const root = document.querySelector('.<?= $uid ?>');
  if (!root || root.dataset.hpbci3dInit === '1') return;
  root.dataset.hpbci3dInit = '1';

  const stage = root.querySelector('.hpb3d-stage');
  const items = Array.from(root.querySelectorAll('.hpb3d-item'));
  const N = items.length;
  if (!stage || !N) return;

  const cfg = {
    rotate: stage.getAttribute('data-rotate') === '1',
    dir: (stage.getAttribute('data-dir') || 'cw') === 'ccw' ? 'ccw' : 'cw',
    speed: Math.max(2, parseInt(stage.getAttribute('data-speed') || '18', 10) || 18),
    pauseHover: stage.getAttribute('data-pause-hover') === '1',
    pauseTap: stage.getAttribute('data-pause-tap') === '1',
    stopIndex: Math.max(0, Math.min(N, parseInt(stage.getAttribute('data-stop-index') || '0', 10) || 0)),
    stopPauseSec: Math.max(0, parseFloat(stage.getAttribute('data-stop-pause') || '0') || 0),
    stopZoom: stage.getAttribute('data-stop-zoom') === '1'
  };

  if (cfg.pauseTap) stage.dataset.pauseTap = "1";

  const baseAngles = items.map((_, i) => (i * 360) / N);

  const stopIdx = cfg.stopIndex > 0 ? (cfg.stopIndex - 1) : -1;
  const stopCard = (stopIdx >= 0 && items[stopIdx]) ? items[stopIdx].querySelector('.hpb3d-card') : null;

  const norm360 = (a) => {
    a = a % 360;
    return a < 0 ? a + 360 : a;
  };

  // Target angle so that stopIndex is in front (0°)
  const targetAngle = (() => {
    if (cfg.stopIndex <= 0) return 0;
    const idx = cfg.stopIndex - 1;
    return norm360(-baseAngles[idx]);
  })();

  let angle = 0;                 // ring angle in degrees (mod 360)
  let stopTriggered = false;

  let pausedManual = false;
  let pausedHover  = false;
  let pausedStop   = false;
  let stopTimer    = null;

  const applyStopZoom = () => {
    if (!cfg.stopZoom || !stopCard) return;
    stopCard.classList.toggle('hpbci-stop-zoom', pausedStop);
  };

  const scheduleStopResume = () => {
    if (stopTimer) {
      clearTimeout(stopTimer);
      stopTimer = null;
    }
    if (cfg.stopPauseSec > 0) {
      stopTimer = setTimeout(() => {
        pausedStop = false;
        applyStopZoom();
      }, Math.round(cfg.stopPauseSec * 1000));
    }
  };

  const isPaused = () => pausedManual || pausedHover || pausedStop;

  const distAlongDir = (from, to, dirSign) => {
    // remaining distance from -> to along direction (dirSign: +1 increasing, -1 decreasing), result in [0..360)
    if (dirSign > 0) return (to - from + 360) % 360;
    return (from - to + 360) % 360;
  };

  const update = () => {
    const styles = getComputedStyle(root);
    const r = parseFloat(styles.getPropertyValue('--hpb3d-radius')) || 260;

    // Use continuous angle for transforms (avoid jitter from mod)
    const aBase = angle;

    for (let i = 0; i < N; i++) {
      const el = items[i];
      const a  = baseAngles[i] + aBase;

      // depth: cos(a) => front: +1, back: -1
      const depth = Math.cos((a * Math.PI) / 180);
      const zIndex = Math.round((depth + 1) * 1000);
      const scale = 0.72 + ((depth + 1) / 2) * 0.28;
      const opacity = 0.35 + ((depth + 1) / 2) * 0.65;

      el.style.zIndex = String(zIndex);
      el.style.opacity = String(opacity);
      el.style.transform = `translate3d(-50%, -50%, 0) rotateY(${a}deg) translateZ(${r}px) rotateY(${-a}deg) scale(${scale})`;
    }
  };

  // If rotation is OFF but stopIndex is set: orient immediately.
  if (!cfg.rotate && cfg.stopIndex > 0) {
    angle = targetAngle;
  }

  // Events
  if (cfg.pauseHover) {
    stage.addEventListener('mouseenter', () => {
      if (!pausedManual && !pausedStop) pausedHover = true;
    }, { passive: true });
    stage.addEventListener('mouseleave', () => {
      pausedHover = false;
    }, { passive: true });
  }

  if (cfg.pauseTap) {
    stage.addEventListener('click', () => {
      // If rotation was auto-stopped, a single click resumes.
      if (pausedStop) {
        pausedStop = false;
        pausedManual = false;
        pausedHover = false;
        stopTriggered = true;
        applyStopZoom();
        return;
      }

      // Normal toggle (pause/resume)
      pausedManual = !pausedManual;
      if (!pausedManual) pausedHover = false;
    });
  }

  // Animation loop
  let last = performance.now();
  const dirSign = cfg.dir === 'ccw' ? 1 : -1;
  const degPerSec = 360 / cfg.speed;
  const EPS = 0.75; // stop tolerance in degrees

  const tick = (now) => {
    const dt = (now - last) / 1000;
    last = now;

    if (cfg.rotate && !isPaused()) {
      const step = dirSign * degPerSec * dt;
      const cur = norm360(angle);
      const next = norm360(angle + step);

      if (cfg.stopIndex > 0 && !stopTriggered) {
        const remaining = distAlongDir(cur, targetAngle, dirSign);
        const stepAbs = Math.abs(step);

        if (remaining <= stepAbs + EPS) {
          angle = targetAngle;     // snap exact
          pausedStop = true;       // stop rotation
          stopTriggered = true;
          scheduleStopResume();
          applyStopZoom();
        } else {
          angle += step;
        }
      } else {
        angle += step;
      }
    }

    update();
    requestAnimationFrame(tick);
  };

  update();
  requestAnimationFrame(tick);
})();
</script>

<?php
// Custom CSS (Advanced tab)
if (!empty($props['css']) && is_string($props['css'])) {
    echo '<style>' . $props['css'] . '</style>';
}

echo $wrap->end();
