<?php
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Router\Route;
use Joomla\Component\Content\Site\Helper\RouteHelper;

$props = $props ?? [];
$attrs = $attrs ?? [];

$debug = !empty($props['debug']);

try {

// UID for scoping
$uid = 'hpbcal-' . substr(md5(json_encode($props) . uniqid('', true)), 0, 12);
$uid = preg_replace('~[^a-zA-Z0-9\-_]~', '', $uid);

// ---------- Helpers ----------
$csvInts = function (string $s): array {
  $out = [];
  foreach (preg_split('~[,\s]+~', trim($s)) as $p) {
    $p = trim($p);
    if ($p === '') continue;
    $v = (int) $p;
    if ($v > 0) $out[] = $v;
  }
  return array_values(array_unique($out));
};

$csvStrings = function (string $s): array {
  $out = [];
  foreach (preg_split('~[,;\n]+~', $s) as $p) {
    $p = strtolower(trim($p));
    if ($p === '') continue;
    $out[] = $p;
  }
  return array_values(array_unique($out));
};

$normKey = function ($s): string {
  $s = trim((string) $s);
  if ($s === '') return '';
  $s = mb_strtolower($s, 'UTF-8');
  $s = preg_replace('~\s+~u', ' ', $s);
  return $s;
};

$normKeyLoose = function ($s): string {
  $s = trim((string) $s);
  if ($s === '') return '';
  $s = mb_strtolower($s, 'UTF-8');
  // remove everything except a-z0-9 to allow fuzzy matching (e.g. "An/Abreise" == "An / Abreise" == "an_abreise")
  $s = preg_replace('~[^a-z0-9]+~u', '', $s);
  return $s;
};

$getField = function (array $fields, string $wanted) use ($normKey, $normKeyLoose) {
  $k = $normKey($wanted);
  if ($k !== '' && array_key_exists($k, $fields)) {
    return $fields[$k];
  }

  $kl = $normKeyLoose($wanted);
  if ($kl !== '') {
    foreach ($fields as $kk => $vv) {
      if ($normKeyLoose((string) $kk) === $kl) {
        return $vv;
      }
    }
  }

  return '';
};


$toScalar = function ($value): string {
  if (is_array($value)) {
    // Best effort: first scalar
    foreach ($value as $v) {
      if (is_scalar($v)) return trim((string) $v);
      if (is_array($v) && isset($v['value']) && is_scalar($v['value'])) return trim((string) $v['value']);
    }
    return '';
  }
  if (is_object($value)) {
    if (isset($value->value) && is_scalar($value->value)) return trim((string) $value->value);
    return '';
  }
  return trim((string) $value);
};

$normalizeColor = function (string $c): string {
  $c = trim($c);
  if ($c === '') return '';
  $lc = strtolower($c);

  if ($lc === 'red' || $lc === 'green') return $lc;

  // #rgb / #rrggbb / #rrggbbaa (also allow without leading #)
  if (preg_match('~^#?[0-9a-f]{3}([0-9a-f]{3})?([0-9a-f]{2})?$~i', $c)) {
    return ($c[0] === '#') ? $c : ('#' . $c);
  }

  // rgb/rgba(...)
  if (preg_match('~^rgba?\([0-9\.\s,%]+\)$~i', $c)) {
    return $c;
  }

  return '';
};

$parseDate = function ($value) use ($toScalar): ?\DateTimeImmutable {
  $value = $value ?? '';
  // Joomla custom fields can store JSON strings depending on field type
  $v = $toScalar($value);
  if ($v === '') return null;

  // JSON wrapper?
  if ((str_starts_with($v, '{') && str_ends_with($v, '}')) || (str_starts_with($v, '[') && str_ends_with($v, ']'))) {
    $decoded = json_decode($v, true);
    if (is_string($decoded)) {
      $v = trim($decoded);
    } elseif (is_array($decoded)) {
      if (isset($decoded['value']) && is_scalar($decoded['value'])) {
        $v = trim((string) $decoded['value']);
      } elseif (isset($decoded[0]) && is_scalar($decoded[0])) {
        $v = trim((string) $decoded[0]);
      }
    }
  }

  // Timestamp?
  if (preg_match('~^[0-9]{10,13}$~', $v)) {
    $ts = (int) $v;
    if ($ts > 2000000000) $ts = (int) round($ts / 1000); // ms
    return (new \DateTimeImmutable('@' . $ts))->setTimezone(new \DateTimeZone('Europe/Berlin'));
  }

  $tz = new \DateTimeZone('Europe/Berlin');

  foreach (['Y-m-d', 'Y-m-d H:i:s', 'Y-m-d\TH:i:sP', 'd.m.Y', 'd.m.Y H:i', 'd.m.Y H:i:s'] as $fmt) {
    $dt = \DateTimeImmutable::createFromFormat($fmt, $v, $tz);
    if ($dt instanceof \DateTimeImmutable) return $dt;
  }

  // Final fallback
  try {
    return new \DateTimeImmutable($v, $tz);
  } catch (\Throwable $e) {
    return null;
  }
;
};

$extractScalars = function ($value): array {
  if ($value === null) return [];
  if (is_array($value)) {
    $out = [];
    foreach ($value as $v) {
      if (is_scalar($v)) $out[] = trim((string) $v);
      elseif (is_array($v) && isset($v['value']) && is_scalar($v['value'])) $out[] = trim((string) $v['value']);
      elseif (is_object($v) && isset($v->value) && is_scalar($v->value)) $out[] = trim((string) $v->value);
    }
    return array_values(array_filter($out, fn($x) => $x !== ''));
  }
  if (is_object($value)) {
    if (isset($value->value) && is_scalar($value->value)) return [trim((string) $value->value)];
    return [];
  }
  return [trim((string) $value)];
};

$parseMultiDates = function ($value) use ($extractScalars, $parseDate): array {
  $vals = $extractScalars($value);
  $out = [];
  foreach ($vals as $v) {
    // allow CSV in a single string
    foreach (preg_split('~[,
;]+~', $v) as $part) {
      $part = trim($part);
      if ($part === '') continue;
      $dt = $parseDate($part);
      if ($dt) $out[] = $dt->setTime(0,0,0);
    }
  }
  // unique by Y-m-d
  $uniq = [];
  foreach ($out as $dt) {
    $k = $dt->format('Y-m-d');
    $uniq[$k] = $dt;
  }
  return array_values($uniq);
};

// ---------- Config ----------
$catIds = $csvInts((string) ($props['cat_ids'] ?? ''));
$includeSubcats = !empty($props['include_subcats']);
// Keep the user-provided IDs separate. If none are provided, we load by category.
$articleIdsInput = $csvInts((string) ($props['article_ids'] ?? ''));
$articleIds = $articleIdsInput;
$limit = (int) ($props['limit'] ?? 200);
$limit = max(1, min(5000, $limit));

$startField = trim((string) ($props['start_field'] ?? ''));
if ($startField === '') { $startField = 'Von'; }
$endField = trim((string) ($props['end_field'] ?? ''));
if ($endField === '') { $endField = 'Bis'; }
$colorField = trim((string) ($props['color_field'] ?? ''));
if ($colorField === '') { $colorField = 'Color'; }
$statusField = trim((string) ($props['status_field'] ?? ''));
if ($statusField === '') { $statusField = 'Status'; }
$labelField = trim((string) ($props['label_field'] ?? ''));

$arrDepField = trim((string) ($props['arrdep_field'] ?? ''));
if ($arrDepField === '') { $arrDepField = 'An/Abreise'; }

$sf = $normKey($startField);
$ef = $normKey($endField);
$cf = $normKey($colorField);
$stf = $normKey($statusField);
$lf = $normKey($labelField);
$adf = $normKey($arrDepField);

$redValues = $csvStrings((string) ($props['red_values'] ?? 'booked,belegt,blocked,occupied'));
$greenValues = $csvStrings((string) ($props['green_values'] ?? 'free,frei,available,open'));

$enableArrDep = !empty($props['enable_arrdep']);
$arrDepValues = $csvStrings((string) ($props['arrdep_values'] ?? 'an/abreise,an-abreise,anreise/abreise,arrival/departure'));
$arrDepMergeOverlap = !empty($props['arrdep_merge_overlap']);

$redColor = $normalizeColor((string) ($props['red_color'] ?? '#E53935')) ?: '#E53935';
$greenColor = $normalizeColor((string) ($props['green_color'] ?? '#43A047')) ?: '#43A047';
$priority = (($props['priority'] ?? 'red_over_green') === 'last_wins') ? 'last_wins' : 'red_over_green';

$months = (int) ($props['months'] ?? 3);
$months = max(1, min(12, $months));

$startMonthStr = trim((string) ($props['start_month'] ?? ''));
$weekStart = (($props['week_start'] ?? 'monday') === 'sunday') ? 'sunday' : 'monday';
$styleMode = in_array(($props['style_mode'] ?? 'fill'), ['fill','border','dot'], true) ? $props['style_mode'] : 'fill';

$cellSize = (int) ($props['cell_size'] ?? 44);
$cellSize = max(26, min(90, $cellSize));
$cellGap = (int) ($props['cell_gap'] ?? 6);
$cellGap = max(0, min(20, $cellGap));
$radius = (int) ($props['radius'] ?? 10);
$radius = max(0, min(24, $radius));

$showLegend = !empty($props['show_legend']);
$legendRed = trim((string) ($props['legend_red'] ?? 'Belegt'));
$legendGreen = trim((string) ($props['legend_green'] ?? 'Frei'));
$legendArrDep = trim((string) ($props['legend_arrdep'] ?? 'An/Abreise'));

$showTooltip = !empty($props['show_tooltip']);
$linkToArticle = !empty($props['link_to_article']);

$cacheMinutes = (int) ($props['cache_minutes'] ?? 15);
$cacheMinutes = max(0, min(1440, $cacheMinutes));

$debug = !empty($props['debug']);

// ---------- Data loading (with simple file caching) ----------
$dbo = Factory::getDbo();
$langTag = (string) Factory::getLanguage()->getTag();
$nowSql = Factory::getDate()->toSql();

// Expand subcategories up-front so caching/invalidation also sees child-category articles.
if ($catIds && $includeSubcats) {
  $all = array_fill_keys($catIds, true);
  $frontier = $catIds;

  while ($frontier) {
    $q = $dbo->getQuery(true)
      ->select($dbo->qn('id'))
      ->from($dbo->qn('#__categories'))
      ->where($dbo->qn('extension') . ' = ' . $dbo->q('com_content'))
      ->where($dbo->qn('published') . ' = 1')
      ->where($dbo->qn('parent_id') . ' IN (' . implode(',', array_map('intval', $frontier)) . ')');
    $dbo->setQuery($q);
    $children = (array) $dbo->loadColumn();

    $new = [];
    foreach ($children as $cid) {
      $cid = (int) $cid;
      if ($cid > 0 && empty($all[$cid])) {
        $all[$cid] = true;
        $new[] = $cid;
      }
    }
    $frontier = $new;
  }

  $catIds = array_map('intval', array_keys($all));
}

// Source signature: used to invalidate cache when articles change (especially in category-only mode).
$getSourceSig = function () use ($dbo, $articleIdsInput, $catIds, $langTag, $nowSql): ?array {

  $maxChangedExpr = "MAX(COALESCE(NULLIF(c.modified, '0000-00-00 00:00:00'), c.created))";

  // Explicit article IDs mode
  if (!empty($articleIdsInput)) {
    $q = $dbo->getQuery(true)
      ->select([
        'MAX(c.id) AS max_id',
        $maxChangedExpr . ' AS max_changed',
        'COUNT(c.id) AS cnt',
      ])
      ->from($dbo->qn('#__content', 'c'))
      ->where('c.id IN (' . implode(',', array_map('intval', $articleIdsInput)) . ')');
    $dbo->setQuery($q);
    $row = (array) $dbo->loadAssoc();
    return [
      'mode' => 'ids',
      'maxId' => (int) ($row['max_id'] ?? 0),
      'maxChanged' => (string) ($row['max_changed'] ?? ''),
      'count' => (int) ($row['cnt'] ?? 0),
    ];
  }

  // Category mode
  if (!empty($catIds)) {
    $q = $dbo->getQuery(true)
      ->select([
        'MAX(c.id) AS max_id',
        $maxChangedExpr . ' AS max_changed',
        'COUNT(c.id) AS cnt',
      ])
      ->from($dbo->qn('#__content', 'c'))
      ->where('c.state = 1')
      ->where('c.catid IN (' . implode(',', array_map('intval', $catIds)) . ')')
      ->where('(c.publish_up IS NULL OR c.publish_up = ' . $dbo->q('0000-00-00 00:00:00') . ' OR c.publish_up <= ' . $dbo->q($nowSql) . ')')
      ->where('(c.publish_down IS NULL OR c.publish_down = ' . $dbo->q('0000-00-00 00:00:00') . ' OR c.publish_down >= ' . $dbo->q($nowSql) . ')')
      ->where('(c.language = ' . $dbo->q('*') . ' OR c.language = ' . $dbo->q($langTag) . ')');
    $dbo->setQuery($q);
    $row = (array) $dbo->loadAssoc();
    return [
      'mode' => 'cats',
      'maxId' => (int) ($row['max_id'] ?? 0),
      'maxChanged' => (string) ($row['max_changed'] ?? ''),
      'count' => (int) ($row['cnt'] ?? 0),
    ];
  }

  return null;
};

$cacheHash = md5(json_encode([
  'catIds' => $catIds,
  'includeSubcats' => $includeSubcats,
  'articleIds' => $articleIds,
  'limit' => $limit,
  'lang' => $langTag,
  'fields' => [$sf,$ef,$cf,$stf,$lf],
  'map' => [$redValues,$greenValues,$redColor,$greenColor,$priority],
  'arrdep' => [$enableArrDep,$arrDepField,$arrDepValues,$arrDepMergeOverlap],
]));

$cacheDir = JPATH_CACHE . '/hpb_availability_calendar';
$cacheFile = $cacheDir . '/hpbcal_' . $cacheHash . '.json';

$debugInfo = [];
$payload = null;
$cacheHit = false;

if ($cacheMinutes > 0 && is_file($cacheFile)) {
  $age = time() - (int) @filemtime($cacheFile);
  if ($age >= 0 && $age <= ($cacheMinutes * 60)) {
    $json = @file_get_contents($cacheFile);
    $tmp = is_string($json) ? json_decode($json, true) : null;
    if (is_array($tmp)) {
      // Invalidate cache when articles in the chosen category change (new booking added, edited, etc.)
      $sigNow = $getSourceSig();
      $sigOld = is_array($tmp['sourceSig'] ?? null) ? $tmp['sourceSig'] : null;

      $invalidated = false;
      // If the stored cache was created by an older version without a signature, rebuild once.
      if (is_array($sigNow) && !is_array($sigOld)) {
        $invalidated = true;
      } elseif (is_array($sigNow) && is_array($sigOld)) {
        $oldId = (int) ($sigOld['maxId'] ?? 0);
        $newId = (int) ($sigNow['maxId'] ?? 0);
        $oldCh = (string) ($sigOld['maxChanged'] ?? '');
        $newCh = (string) ($sigNow['maxChanged'] ?? '');
        $oldCnt = (int) ($sigOld['count'] ?? 0);
        $newCnt = (int) ($sigNow['count'] ?? 0);

        if ($oldId !== $newId || $oldCh !== $newCh || $oldCnt !== $newCnt) {
          $invalidated = true;
        }
      }

      if (!$invalidated) {
        $payload = $tmp;
        $cacheHit = true;
      } elseif ($debug) {
        $debugInfo['cache_invalidated'] = [
          'sig_old' => $sigOld,
          'sig_now' => $sigNow,
        ];
      }
    }
  }
}

if ($debug) {
  $debugInfo['cache'] = [
    'enabled' => ($cacheMinutes > 0),
    'minutes' => $cacheMinutes,
    'hit' => $cacheHit,
    'file' => basename($cacheFile),
  ];
}


if (!is_array($payload)) {
  // Load articles if not specified directly
  if (!$articleIds && $catIds) {
    $q = $dbo->getQuery(true)
      ->select([
        'c.id', 'c.title', 'c.alias', 'c.catid', 'c.language'
      ])
      ->from($dbo->qn('#__content', 'c'))
      ->where('c.state = 1')
      ->where('c.catid IN (' . implode(',', array_map('intval', $catIds)) . ')')
      ->where('(c.publish_up IS NULL OR c.publish_up = ' . $dbo->q('0000-00-00 00:00:00') . ' OR c.publish_up <= ' . $dbo->q($nowSql) . ')')
      ->where('(c.publish_down IS NULL OR c.publish_down = ' . $dbo->q('0000-00-00 00:00:00') . ' OR c.publish_down >= ' . $dbo->q($nowSql) . ')')
      ->where('(c.language = ' . $dbo->q('*') . ' OR c.language = ' . $dbo->q($langTag) . ')')
      ->order('c.publish_up DESC, c.id DESC');
    $dbo->setQuery($q, 0, $limit);
    $rows = (array) $dbo->loadAssocList();
    $articleIds = array_values(array_filter(array_map(fn($r) => (int) ($r['id'] ?? 0), $rows)));
  }

  // Load minimal article meta for links/debug
  $meta = [];
  if ($articleIds) {
    $q = $dbo->getQuery(true)
      ->select(['c.id','c.title','c.alias','c.catid'])
      ->from($dbo->qn('#__content','c'))
      ->where('c.id IN (' . implode(',', array_map('intval',$articleIds)) . ')');
    $dbo->setQuery($q);
    foreach ((array) $dbo->loadAssocList() as $r) {
      $id = (int) ($r['id'] ?? 0);
      if ($id <= 0) continue;
      $meta[$id] = [
        'id' => $id,
        'title' => (string) ($r['title'] ?? ''),
        'alias' => (string) ($r['alias'] ?? ''),
        'catid' => (int) ($r['catid'] ?? 0),
      ];
    }
  }

  // Load ALL custom fields for those items (robust against title/name confusion)
  $byItem = [];
  $byItemRaw = []; // debug listing of available fields
  if ($articleIds) {
    $q = $dbo->getQuery(true)
      ->select(['fv.item_id', 'f.context', 'f.name', 'f.title', 'fv.value'])
      ->from($dbo->qn('#__fields_values', 'fv'))
      ->join('INNER', $dbo->qn('#__fields', 'f') . ' ON f.id = fv.field_id')
      ->where('(f.context LIKE ' . $dbo->q('com_content.%') . ')')
      ->where('fv.item_id IN (' . implode(',', array_map('intval', $articleIds)) . ')');
    $dbo->setQuery($q);
    $vals = (array) $dbo->loadAssocList();

    foreach ($vals as $v) {
      $iid = (int) ($v['item_id'] ?? 0);
      $name = (string) ($v['name'] ?? '');
      $title = (string) ($v['title'] ?? '');
      $ctx = (string) ($v['context'] ?? '');
      if ($iid <= 0) continue;

      $val = $v['value'] ?? '';

      $kn = $normKey($name);
      if ($kn !== '') {
        $byItem[$iid][$kn] = $val;
        $byItemRaw[$iid][$kn] = ['context' => $ctx, 'name' => $name, 'title' => $title, 'value' => $val];
      }

      $kt = $normKey($title);
      if ($kt !== '' && $kt !== $kn) {
        $byItem[$iid][$kt] = $val;
        $byItemRaw[$iid][$kt] = ['context' => $ctx, 'name' => $name, 'title' => $title, 'value' => $val];
      }
    }
  }

  // Map articles -> day colors
  $dayMap = [];

  $isRedish = function (string $roleOrColor) use ($redColor): bool {
    $c = strtolower(trim($roleOrColor));
    if ($c === 'red') return true;
    return (trim($roleOrColor) === $redColor);
  };
  $isGreenish = function (string $roleOrColor) use ($greenColor): bool {
    $c = strtolower(trim($roleOrColor));
    if ($c === 'green') return true;
    return (trim($roleOrColor) === $greenColor);
  };

  foreach ($articleIds as $id) {

    $fields = (array) ($byItem[$id] ?? []);

    $startRaw  = $startField !== '' ? $getField($fields, $startField) : '';
    $endRaw    = $endField !== '' ? $getField($fields, $endField) : '';
    $colorRaw  = $colorField !== '' ? $getField($fields, $colorField) : '';
    $statusRaw = $statusField !== '' ? $getField($fields, $statusField) : '';
    $labelRaw  = $labelField !== '' ? $getField($fields, $labelField) : '';
    $arrDepRaw = ($enableArrDep && $arrDepField !== '') ? $getField($fields, $arrDepField) : '';
$start = $parseDate($startRaw);
    if (!$start) {
      if ($debug) {
        $debugInfo['skipped'][] = [
          'article_id' => $id,
          'title' => $meta[$id]['title'] ?? '',
          'reason' => 'Start date not found or unparsable',
          'configured_start_key' => $startField,
          'available_keys' => array_keys((array) ($byItemRaw[$id] ?? [])),
          'raw_start_value' => $startRaw,
        ];
      }
      continue;
    }

    $end = $parseDate($endRaw) ?: $start;

    // normalize to date-only
    $s = $start->setTime(0,0,0);
    $e = $end->setTime(0,0,0);

    if ($e < $s) {
      $tmp = $s; $s = $e; $e = $tmp;
    }

    // Determine color + role
    $role = 'custom';
    $color = '';

    $st = strtolower($toScalar($statusRaw));

    // Arrival/Departure (split day):
    // 1) Preferred: a dedicated date field (arrdep_field) containing one or multiple dates
    // 2) Legacy fallback: Status matches arrdep_values (then the whole range is rendered as split)
    $arrDepDates = ($enableArrDep && $arrDepField !== '') ? $parseMultiDates($arrDepRaw) : [];
    $forceArrDepWholeRange = ($enableArrDep && $st !== '' && in_array($st, $arrDepValues, true) && !$arrDepDates);

    if ($forceArrDepWholeRange) {
      $role = 'arrdep';
      $color = $redColor; // placeholder (actual split rendered via CSS vars)
    } else {
      $colorRawS = $toScalar($colorRaw);
      if ($colorRawS !== '') {
        $nc = $normalizeColor($colorRawS);
        if ($nc === 'red') { $role = 'red'; $color = $redColor; }
        elseif ($nc === 'green') { $role = 'green'; $color = $greenColor; }
        elseif ($nc !== '') { $color = $nc; }
      }

      if ($color === '') {
        if ($st !== '') {
          if (in_array($st, $redValues, true)) { $role = 'red'; $color = $redColor; }
          elseif (in_array($st, $greenValues, true)) { $role = 'green'; $color = $greenColor; }
        }
      }
    }

    $tooltip = $toScalar($labelRaw);
    if ($tooltip === '') $tooltip = (string) ($meta[$id]['title'] ?? '');

    $link = null;
    if ($linkToArticle && !empty($meta[$id])) {
      $a = $meta[$id];
      $slug = $a['id'] . ':' . $a['alias'];
      $catslug = $a['catid'];
      $link = Route::_(RouteHelper::getArticleRoute($slug, $catslug, $a['language'] ?? '*'));
    }

    // Add each day in range, inclusive
    if ($color !== '' || $forceArrDepWholeRange) {
    for ($d = $s; $d <= $e; $d = $d->modify('+1 day')) {
      $key = $d->format('Y-m-d');

      if (!isset($dayMap[$key])) {
        $dayMap[$key] = [
          'color' => $color,
          'role' => $role,
          'tooltip' => $tooltip,
          'link' => $link,
        ];
        continue;
      }


      $old = $dayMap[$key];

      // Arrival/Departure split day always wins
      if ($enableArrDep) {
        if (($old['role'] ?? '') === 'arrdep') {
          continue;
        }

        if ($role === 'arrdep') {
          $dayMap[$key] = [
            'color' => $color,
            'role' => $role,
            'tooltip' => $tooltip,
            'link' => $link,
          ];
          continue;
        }
      }

      // Determine red/green for overlap + priority rules
      $oldIsGreen = ($old['role'] ?? '') === 'green' || $isGreenish((string) ($old['color'] ?? ''));
      $oldIsRed = ($old['role'] ?? '') === 'red' || $isRedish((string) ($old['color'] ?? ''));

      $newIsGreen = ($role === 'green') || $isGreenish($color);
      $newIsRed = ($role === 'red') || $isRedish($color);

      // Optional: if both red and green land on the same day -> split (triangle)
      if ($enableArrDep && $arrDepMergeOverlap && (($oldIsRed && $newIsGreen) || ($oldIsGreen && $newIsRed))) {
        $mergedTooltip = (string) ($old['tooltip'] ?? '');
        if ($mergedTooltip === '') {
          $mergedTooltip = $tooltip;
        } elseif ($tooltip !== '' && $tooltip !== $mergedTooltip) {
          $mergedTooltip .= ' / ' . $tooltip;
        }

        $dayMap[$key] = [
          'color' => $redColor,
          'role' => 'arrdep',
          'tooltip' => $mergedTooltip,
          'link' => $link ?: ($old['link'] ?? null),
        ];
        continue;
      }

      if ($priority === 'last_wins') {
        $dayMap[$key] = [
          'color' => $color,
          'role' => $role,
          'tooltip' => $tooltip,
          'link' => $link,
        ];
        continue;
      }

      // red overrides green
      if ($newIsRed && $oldIsGreen) {
        $dayMap[$key] = [
          'color' => $color,
          'role' => $role,
          'tooltip' => $tooltip,
          'link' => $link,
        ];
        continue;
      }

      if ($oldIsRed && $newIsGreen) {
        continue; // keep old red
      }

      // otherwise last wins
      $dayMap[$key] = [
        'color' => $color,
        'role' => $role,
        'tooltip' => $tooltip,
        'link' => $link,
      ];
    }

    }

    // Apply Arrival/Departure date overrides (arrdep_field)
    if ($enableArrDep && !empty($arrDepDates)) {
      foreach ($arrDepDates as $adt) {
        $k = $adt->format('Y-m-d');

        if (isset($dayMap[$k]) && (($dayMap[$k]['role'] ?? '') === 'arrdep')) {
          // merge tooltip if different
          $oldTip = (string) ($dayMap[$k]['tooltip'] ?? '');
          if ($tooltip !== '' && $tooltip !== $oldTip) {
            $dayMap[$k]['tooltip'] = $oldTip === '' ? $tooltip : ($oldTip . ' / ' . $tooltip);
          }
          // prefer a link if present
          if (!empty($link) && empty($dayMap[$k]['link'])) {
            $dayMap[$k]['link'] = $link;
          }
          continue;
        }

        $dayMap[$k] = [
          'color' => $redColor,
          'role' => 'arrdep',
          'tooltip' => $tooltip,
          'link' => $link,
        ];
      }
    }

    if ($debug) {
      $debugInfo['resolved'][] = [
        'article_id' => $id,
        'title' => $meta[$id]['title'] ?? '',
        'start_raw' => $toScalar($startRaw),
        'end_raw' => $toScalar($endRaw),
        'start' => $s->format('Y-m-d'),
        'end' => $e->format('Y-m-d'),
        'color_raw' => $toScalar($colorRaw),
        'status_raw' => $toScalar($statusRaw),
        'arrdep_raw' => $toScalar($arrDepRaw),
        'arrdep_dates' => array_map(fn($d) => $d->format('Y-m-d'), (array) ($arrDepDates ?? [])),
        'role' => $role,
        'color' => $color,
        'tooltip' => $tooltip,
      ];
    }
  }

  $payload = [
    'dayMap' => $dayMap,
    'articleCount' => count($articleIds),
    'articleIds' => $articleIds,
    'metaCount' => count($meta),
    'availableKeysSample' => $articleIds ? array_slice(array_keys((array) ($byItemRaw[$articleIds[0]] ?? [])), 0, 50) : [],
    'availableContextsSample' => $articleIds ? array_values(array_unique(array_filter(array_map(function ($x) { return (string) ($x['context'] ?? ''); }, array_slice((array) ($byItemRaw[$articleIds[0]] ?? []), 0, 50))))) : [],
    // Used to bust cache automatically when category articles change.
    'sourceSig' => $getSourceSig(),
  ];

  // Store cache (best-effort)
  if ($cacheMinutes > 0) {
    if (!is_dir($cacheDir)) {
      @mkdir($cacheDir, 0755, true);
    }
    @file_put_contents($cacheFile, json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES), LOCK_EX);
  }
}

$articleIds = (array) ($payload['articleIds'] ?? $articleIds);
$dayMap = (array) ($payload['dayMap'] ?? []);
$articleCount = (int) ($payload['articleCount'] ?? 0);

// ---------- Month generation ----------
$tz = new \DateTimeZone('Europe/Berlin');

if ($startMonthStr !== '') {
  $dt = $parseDate($startMonthStr);
  $startMonth = ($dt ?: new \DateTimeImmutable('now', $tz))->modify('first day of this month');
} else {
  $startMonth = (new \DateTimeImmutable('now', $tz))->modify('first day of this month');
}

$monthsList = [];
for ($i = 0; $i < $months; $i++) {
  $monthsList[] = $startMonth->modify('+' . $i . ' month');
}

$weekdayLabelsMon = ['Mo','Di','Mi','Do','Fr','Sa','So'];
$weekdayLabelsSun = ['So','Mo','Di','Mi','Do','Fr','Sa'];
$weekdayLabels = ($weekStart === 'sunday') ? $weekdayLabelsSun : $weekdayLabelsMon;

// ---------- Output ----------
$debugBlock = '';
if ($debug) {
  $debugView = [
    'cache' => (array) ($debugInfo['cache'] ?? []),
    'cache_invalidated' => (array) ($debugInfo['cache_invalidated'] ?? []),
    'articleCount' => $articleCount,
    'cat_ids' => $catIds,
    'article_ids' => $articleIds,
    'mapping' => [
      'start_field' => $startField,
      'end_field' => $endField,
      'color_field' => $colorField,
      'status_field' => $statusField,
      'label_field' => $labelField,
      'arrdep_field' => $arrDepField,
      'normalized' => [$sf,$ef,$cf,$stf,$lf,$adf],
    ],
    'payload' => [
      'metaCount' => (int) ($payload['metaCount'] ?? 0),
      'availableKeysSample' => (array) ($payload['availableKeysSample'] ?? []),
      'availableContextsSample' => (array) ($payload['availableContextsSample'] ?? []),
    ],
    'resolved_first_10' => array_slice((array) ($debugInfo['resolved'] ?? []), 0, 10),
    'skipped_first_10' => array_slice((array) ($debugInfo['skipped'] ?? []), 0, 10),
  ];

  $debugBlock = '<pre class="hpbcal-debug">' . htmlspecialchars(json_encode($debugView, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), ENT_QUOTES, 'UTF-8') . '</pre>';
}
?>
<div class="<?= htmlspecialchars($uid, ENT_QUOTES) ?> hpbcal" data-element="hpb_availability_calendar">
  <style>
  .<?= $uid ?>{
    --hpbcal-cell: <?= (int) $cellSize ?>px;
    --hpbcal-gap: <?= (int) $cellGap ?>px;
    --hpbcal-radius: <?= (int) $radius ?>px;
    --hpbcal-red: <?= htmlspecialchars($redColor, ENT_QUOTES) ?>;
    --hpbcal-green: <?= htmlspecialchars($greenColor, ENT_QUOTES) ?>;
  }

  .<?= $uid ?> .hpbcal-wrap{
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
    gap: 22px;
    align-items: start;
  }

  .<?= $uid ?> .hpbcal-month{
    padding: 14px 14px 12px;
    border: 1px solid rgba(0,0,0,.08);
    border-radius: 16px;
    background: rgba(255,255,255,.02);
  }

  .<?= $uid ?> .hpbcal-title{
    font-weight: 700;
    margin: 0 0 10px 0;
    line-height: 1.2;
  }

  .<?= $uid ?> .hpbcal-grid{
    display: grid;
    grid-template-columns: repeat(7, var(--hpbcal-cell));
    gap: var(--hpbcal-gap);
  }

  .<?= $uid ?> .hpbcal-wd{
    font-size: 12px;
    opacity: .75;
    text-align: center;
    height: 18px;
  }

  .<?= $uid ?> .hpbcal-day{
    width: var(--hpbcal-cell);
    height: var(--hpbcal-cell);
    display: grid;
    place-items: center;
    border-radius: var(--hpbcal-radius);
    border: 1px solid rgba(0,0,0,.10);
    font-size: 13px;
    user-select: none;
    position: relative;
    overflow: hidden;
    background: rgba(0,0,0,.02);
    text-decoration: none;
    color: inherit;
  }

  .<?= $uid ?> .hpbcal-day.is-empty{
    border: 1px solid transparent;
    background: transparent;
  }

  .<?= $uid ?> .hpbcal-day.is-colored.fill{
    background: var(--hpbcal-color);
    border-color: transparent;
    color: #fff;
  }

  .<?= $uid ?> .hpbcal-day.is-colored.border{
    background: transparent;
    border-color: var(--hpbcal-color);
  }

  .<?= $uid ?> .hpbcal-day.is-colored.dot::after{
    content: "";
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: var(--hpbcal-color);
    position: absolute;
    bottom: 6px;
    left: 50%;
    transform: translateX(-50%);
  }

  .<?= $uid ?> .hpbcal-day.is-arrdep{
    background: linear-gradient(135deg, var(--hpbcal-red) 0 50%, var(--hpbcal-green) 50% 100%);
    border-color: transparent;
    color: #fff;
  }


  /* Ensure split day overrides fill/border/dot display modes */
  .<?= $uid ?> .hpbcal-day.is-arrdep.is-colored.fill,
  .<?= $uid ?> .hpbcal-day.is-arrdep.is-colored.border,
  .<?= $uid ?> .hpbcal-day.is-arrdep.is-colored.dot{
    background: linear-gradient(135deg, var(--hpbcal-red) 0 50%, var(--hpbcal-green) 50% 100%);
    border-color: transparent;
    color: #fff;
  }
  .<?= $uid ?> .hpbcal-day.is-arrdep.dot::after{
    display: none;
  }

  .<?= $uid ?> .hpbcal-swatch.is-arrdep{
    background: linear-gradient(135deg, var(--hpbcal-red) 0 50%, var(--hpbcal-green) 50% 100%);
  }

  .<?= $uid ?> .hpbcal-legend{
    display: flex;
    gap: 12px;
    align-items: center;
    margin-top: 12px;
    font-size: 13px;
  }

  .<?= $uid ?> .hpbcal-legend-item{
    display: inline-flex;
    align-items: center;
    gap: 8px;
    opacity: .95;
  }

  .<?= $uid ?> .hpbcal-swatch{
    width: 14px;
    height: 14px;
    border-radius: 4px;
    background: var(--hpbcal-color);
    border: 1px solid rgba(0,0,0,.15);
  }

  .<?= $uid ?> .hpbcal-debug{
    margin: 0 0 18px 0;
    padding: 12px;
    border-radius: 12px;
    background: rgba(0,0,0,.05);
    font-size: 12px;
    overflow: auto;
    max-height: 420px;
  }
  </style>

  <?= $debugBlock ?>

  <div class="hpbcal-wrap">
    <?php foreach ($monthsList as $m): ?>
      <?php
        $monthLabel = $m->format('F Y');
        $first = $m;
        $daysInMonth = (int) $first->format('t');

        // 0..6 offset
        $w = (int) $first->format('N'); // 1=Mon..7=Sun
        $offset = ($weekStart === 'monday') ? ($w - 1) : ($w % 7); // Sunday -> 0

        $cells = [];
        for ($i = 0; $i < $offset; $i++) $cells[] = null;
        for ($d = 1; $d <= $daysInMonth; $d++) $cells[] = $d;

        // Pad to full weeks
        while (count($cells) % 7 !== 0) $cells[] = null;
      ?>
      <div class="hpbcal-month">
        <div class="hpbcal-title"><?= htmlspecialchars($monthLabel, ENT_QUOTES, 'UTF-8') ?></div>

        <div class="hpbcal-grid">
          <?php foreach ($weekdayLabels as $wd): ?>
            <div class="hpbcal-wd"><?= htmlspecialchars($wd, ENT_QUOTES, 'UTF-8') ?></div>
          <?php endforeach; ?>

          <?php foreach ($cells as $d): ?>
            <?php if ($d === null): ?>
              <div class="hpbcal-day is-empty"></div>
            <?php else: ?>
              <?php
                $date = $m->setDate((int) $m->format('Y'), (int) $m->format('m'), (int) $d);
                $key = $date->format('Y-m-d');
                $info = $dayMap[$key] ?? null;
                $role = is_array($info) ? (string) ($info['role'] ?? '') : '';
                $isColored = is_array($info) && (!empty($info['color']) || $role === 'arrdep');
                $color = $isColored ? (string) ($info['color'] ?? '') : '';
                $tooltip = $isColored ? (string) ($info['tooltip'] ?? '') : '';
                $link = $isColored ? ($info['link'] ?? null) : null;

                $classes = 'hpbcal-day';
                if ($isColored) $classes .= ' is-colored ' . $styleMode;
                if ($role === 'arrdep') $classes .= ' is-arrdep';
              ?>
              <?php if ($link): ?>
                <a class="<?= htmlspecialchars($classes, ENT_QUOTES) ?>" href="<?= htmlspecialchars((string) $link, ENT_QUOTES) ?>"
                   <?php if ($showTooltip && $tooltip !== ''): ?>title="<?= htmlspecialchars($tooltip, ENT_QUOTES, 'UTF-8') ?>"<?php endif; ?>
                   style="<?php if ($isColored): ?>--hpbcal-color: <?= htmlspecialchars($color, ENT_QUOTES) ?>;<?php endif; ?>">
                  <?= (int) $d ?>
                </a>
              <?php else: ?>
                <div class="<?= htmlspecialchars($classes, ENT_QUOTES) ?>"
                   <?php if ($showTooltip && $tooltip !== ''): ?>title="<?= htmlspecialchars($tooltip, ENT_QUOTES, 'UTF-8') ?>"<?php endif; ?>
                   style="<?php if ($isColored): ?>--hpbcal-color: <?= htmlspecialchars($color, ENT_QUOTES) ?>;<?php endif; ?>">
                  <?= (int) $d ?>
                </div>
              <?php endif; ?>
            <?php endif; ?>
          <?php endforeach; ?>
        </div>

        <?php if ($showLegend): ?>
          <div class="hpbcal-legend">
            <span class="hpbcal-legend-item" style="--hpbcal-color: <?= htmlspecialchars($redColor, ENT_QUOTES) ?>;">
              <span class="hpbcal-swatch"></span>
              <span><?= htmlspecialchars($legendRed, ENT_QUOTES, 'UTF-8') ?></span>
            </span>
            <span class="hpbcal-legend-item" style="--hpbcal-color: <?= htmlspecialchars($greenColor, ENT_QUOTES) ?>;">
              <span class="hpbcal-swatch"></span>
              <span><?= htmlspecialchars($legendGreen, ENT_QUOTES, 'UTF-8') ?></span>
            </span>
            <?php if ($enableArrDep): ?>
              <span class="hpbcal-legend-item">
                <span class="hpbcal-swatch is-arrdep"></span>
                <span><?= htmlspecialchars($legendArrDep, ENT_QUOTES, 'UTF-8') ?></span>
              </span>
            <?php endif; ?>
          </div>
        <?php endif; ?>
      </div>
    <?php endforeach; ?>
  </div>

  <?php if (!empty($props['css'])): ?>
    <style><?= $props['css'] ?></style>
  <?php endif; ?>

</div>

<?php
} catch (\Throwable $e) {
  if (!empty($debug)) {
    $msg = htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    $file = htmlspecialchars(basename((string) $e->getFile()), ENT_QUOTES, 'UTF-8');
    $line = (int) $e->getLine();
    echo '<div class="uk-alert uk-alert-danger"><strong>HPB Availability Calendar error</strong><br>' . $msg . '<br><small>' . $file . ':' . $line . '</small></div>';
  }
}