<?php
defined('_JEXEC') or die;

$props = $props ?? [];
$attrs = $attrs ?? [];

// Unique id per render
$uid = 'ifp-' . substr(md5(json_encode($props) . uniqid('', true)), 0, 12);
$uid = preg_replace('~[^a-zA-Z0-9\-_]~', '', $uid);

// Helpers
$pickSrc = function ($img): string {

  if (is_array($img)) {
    foreach (['src','url','path','image','value'] as $k) {
      if (!empty($img[$k]) && is_string($img[$k])) return trim((string) $img[$k]);
    }
    if (!empty($img[0]) && is_array($img[0]) && !empty($img[0]['src'])) return trim((string) $img[0]['src']);
    return '';
  }

  if (is_object($img)) {
    foreach (['src','url','path','image','value'] as $k) {
      if (isset($img->$k) && is_string($img->$k) && trim((string) $img->$k) !== '') return trim((string) $img->$k);
    }
    if (method_exists($img, '__toString')) return trim((string) $img);
    return '';
  }

  if (is_string($img)) return trim($img);

  return '';
};

$src = $pickSrc($props['image'] ?? null);
if ($src === '') return;

// Normalize image URL (spaces, relative paths)
$src = str_replace('\\', '/', $src);
if (!preg_match('~^(https?:)?//~i', $src) && $src[0] !== '/') {
  $src = '/' . ltrim($src, '/');
}
$src = str_replace(' ', '%20', $src);

$alt = trim((string) ($props['alt'] ?? ''));

// Height
$height = (int) ($props['height'] ?? 120);
$height = max(10, $height);

// Margins (spacers)
$mt  = (float) ($props['margin_top'] ?? 0);
$mb  = (float) ($props['margin_bottom'] ?? 0);
$mtS = (float) ($props['margin_top_s'] ?? $mt);
$mbS = (float) ($props['margin_bottom_s'] ?? $mb);
$mt  = max(0, min(10000, $mt));
$mb  = max(0, min(10000, $mb));
$mtS = max(0, min(10000, $mtS));
$mbS = max(0, min(10000, $mbS));

// ViewBox
$viewbox = trim((string) ($props['viewbox'] ?? '0 0 1200 120'));
if (!preg_match('~^\s*[-0-9.]+\s+[-0-9.]+\s+[0-9.]+\s+[0-9.]+\s*$~', $viewbox)) {
  $viewbox = '0 0 1200 120';
}

// Path input
$pathInput = trim((string) ($props['path_input'] ?? ''));
$pathMode  = (string) ($props['path_mode'] ?? 'auto');

$extractD = function (string $input): string {
  $in = trim($input);
  if ($in === '') return '';
  if (preg_match('~d\s*=\s*("([^"]+)"|\'([^\']+)\')~i', $in, $m)) {
    return (string) ($m[2] ?? $m[3] ?? '');
  }
  return $in;
};

$sanitizeD = function (string $d): string {
  $d = trim($d);
  if ($d === '') return '';
  // Allow only SVG path chars
  $d = preg_replace('~[^MmZzLlHhVvCcSsQqTtAa0-9eE\.\,\-\+\s]~', '', $d);
  $d = preg_replace('~\s+~', ' ', $d);
  return trim($d);
};

// Derive a "curve only" motion path for typical M..V..C..V..Z shapes
$deriveCurvePath = function (string $d): string {
  $dn = trim($d);
  if ($dn === '') return '';

  // Match: M0,0 V60 C... V0 Z  => start at first V point, then curve(s)
  if (preg_match('~^\s*M\s*([-0-9.]+)\s*,\s*([-0-9.]+)\s*V\s*([-0-9.]+)\s*(C.*)\s*V\s*([-0-9.]+)\s*Z\s*$~i', $dn, $m)) {
    $x = $m[1];
    $y = $m[3];
    $curves = trim($m[4]);
    return 'M' . $x . ',' . $y . ' ' . $curves;
  }

  // If it's already an open cubic curve path, accept it
  if (preg_match('~^\s*M~', $dn) && preg_match('~C~', $dn) && !preg_match('~Z~i', $dn)) {
    return $dn;
  }

  return $dn;
};

$dFull = $sanitizeD($extractD($pathInput));
if ($dFull === '') return;

$motionD = $dFull;
if ($pathMode === 'curve') {
  $motionD = $sanitizeD($deriveCurvePath($dFull));
}

// Path visuals
$showPath = !empty($props['show_path']);
$fill     = (string) ($props['path_fill'] ?? 'none');
$stroke   = (string) ($props['path_stroke'] ?? '#141316');
$strokeW  = (float) ($props['path_stroke_width'] ?? 2);
$opacityP = (float) ($props['path_opacity'] ?? 0.35);
$strokeW  = max(0, $strokeW);
$opacityP = max(0, min(1, $opacityP));

// Image size
$imgW = (float) ($props['img_width'] ?? 0);
if ($imgW <= 0) {
  $imgW = (float) ($props['img_w'] ?? 80);
}
$imgH = (float) ($props['img_height'] ?? 0);
if ($imgH <= 0) {
  $imgH = (float) ($props['img_h'] ?? 80);
}
$imgW = max(1, $imgW);
$imgH = max(1, $imgH);


// Animation settings
$dur   = (float) ($props['dur'] ?? 24);
$delay = (float) ($props['delay'] ?? 0);
$dur   = max(0.2, $dur);
$delay = max(0.0, $delay);

$pauseBetween = (float) ($props['pause_between'] ?? 0);
$pauseBetween = max(0.0, $pauseBetween);
$totalDur = $dur + $pauseBetween;

$repeat = (string) ($props['repeat'] ?? 'indefinite');
if (!in_array($repeat, ['indefinite','1','2','3','4','5'], true)) $repeat = 'indefinite';
$isLooping = ($repeat === 'indefinite' || ((int) $repeat) > 1);

$playback = (string) ($props['playback'] ?? 'loop');
if (!in_array($playback, ['loop','pingpong'], true)) $playback = 'loop';

$hideReturn = !empty($props['hide_return']) && $playback === 'pingpong';

$direction = (string) ($props['direction'] ?? 'normal');
if (!in_array($direction, ['normal','reverse'], true)) $direction = 'normal';

$rotate = !empty($props['rotate']) ? 'auto' : '0';

$pauseHover = !empty($props['pause_hover']);
$pauseTap   = !empty($props['pause_tap']);

// Fade settings (optional)
$fadeEnable = !empty($props['fade_enable']);
$fadeInSec  = (float) ($props['fade_in'] ?? 0.5);
$fadeOutSec = (float) ($props['fade_out'] ?? 0.5);
$fadeInSec  = max(0.0, $fadeInSec);
$fadeOutSec = max(0.0, $fadeOutSec);

// Start offset (margin-left) at cycle start
$startMarginLeft = (float) ($props['start_margin_left'] ?? 0);
$startMarginLeft = abs($startMarginLeft);

$enterTime = (float) ($props['enter_time'] ?? 0.35);
$enterTime = max(0.0, $enterTime);

// X/Y base position (centered)
$xNorm = -$imgW / 2;
$y = -$imgH / 2;

// Apply start offset to the base x so the image starts off-screen already
$xBase = $xNorm;
if ($startMarginLeft > 0.0001) {
  $xBase = $xNorm - $startMarginLeft;
}

// Build motion timing fractions
$fForwardEnd = 1.0;
$fReturnEnd  = 1.0;

if ($playback === 'pingpong') {
  if ($pauseBetween > 0.0001) {
    $fForwardEnd = ($dur / 2) / $totalDur;
    $fReturnEnd  = $dur / $totalDur;
  } else {
    $fForwardEnd = 0.5;
    $fReturnEnd  = 1.0;
  }
}

// Animation helper
$fmt = static function (float $t): string {
  $t = max(0.0, min(1.0, $t));
  $s = rtrim(rtrim(sprintf('%.6F', $t), '0'), '.');
  return $s === '' ? '0' : $s;
};

$addPoint = static function (array &$times, array &$vals, float $t, string $v) use ($fmt): void {
  $ts = $fmt($t);
  if (!empty($times) && $times[count($times) - 1] === $ts) {
    $vals[count($vals) - 1] = $v;
    return;
  }
  $times[] = $ts;
  $vals[]  = $v;
};

// Decide if we need opacity animation (to ensure offscreen start and hide-return work, and to mask repeat resets)
$needsOpacity = $fadeEnable || $hideReturn || $isLooping || ($startMarginLeft > 0.0001);

// Build opacity timeline
$opacityAnim = null;
if ($needsOpacity) {

  $autoMask = 0.02; // 20ms mask to avoid loop/reset flashes
  $inSec = $fadeEnable ? $fadeInSec : 0.0;
  $outSec = $fadeEnable ? $fadeOutSec : 0.0;

  if ($inSec < 0.0001 && ($startMarginLeft > 0.0001)) $inSec = max($autoMask, min($enterTime, 0.2));
  if ($outSec < 0.0001 && ($hideReturn || $isLooping)) $outSec = $autoMask;

  $inFrac  = min(0.5, $inSec / $totalDur);
  $outFrac = min(0.5, $outSec / $totalDur);

  $times = [];
  $vals  = [];

  $visibleEnd = 1.0;
  if ($hideReturn) $visibleEnd = $fForwardEnd;

  // Fade in
  $tInEnd = max(0.0, min($visibleEnd, $inFrac));
  if ($tInEnd < 0.0001 && $visibleEnd > 0.0001) $tInEnd = min(0.0001, $visibleEnd);

  // Fade out start (only if we need to end hidden: hideReturn, or looping)
  $endHidden = $hideReturn || $isLooping;
  $tOutStart = $visibleEnd;
  if ($endHidden && $outFrac > 0.00001) {
    $tOutStart = max($tInEnd, $visibleEnd - $outFrac);
  }

  // Start hidden
  $addPoint($times, $vals, 0.0, '0');
  $addPoint($times, $vals, $tInEnd, '1');

  // Stay visible until fade out
  if ($tOutStart > $tInEnd) {
    $addPoint($times, $vals, $tOutStart, '1');
  }

  // Fade out to hidden at visibleEnd (if needed)
  if ($endHidden) {
    $addPoint($times, $vals, $visibleEnd, '0');
  } else {
    $addPoint($times, $vals, $visibleEnd, '1');
  }

  // If hideReturn, remain hidden for rest of cycle
  if ($visibleEnd < 1.0) {
    $addPoint($times, $vals, 1.0, '0');
  } else {
    // Looping: ensure end at 0 to mask restart
    if ($isLooping) {
      $addPoint($times, $vals, 1.0, '0');
    } else {
      $addPoint($times, $vals, 1.0, '1');
    }
  }

  $opacityAnim = [
    'values'   => implode(';', $vals),
    'keyTimes' => implode(';', $times),
  ];
}

// Build x animation for start offset
$xAnim = null;
if ($startMarginLeft > 0.0001) {
  $eps = 0.0001;

  $enterFrac = 0.0;
  if ($totalDur > 0) {
    $enterFrac = min(0.5, $enterTime / $totalDur);
  }
  // Make sure enter happens during the visible part (for hideReturn pingpong)
  $limit = $hideReturn ? max(0.0001, $fForwardEnd - $eps) : 1.0;
  $enterFrac = max(0.0, min($enterFrac, $limit));

  $times = [];
  $vals  = [];

  $addPoint($times, $vals, 0.0, (string) $xBase);
  $addPoint($times, $vals, max($eps, $enterFrac), (string) $xNorm);

  // For pingpong with pause at start: reset to offscreen during pause (invisible anyway)
  if ($hideReturn && $playback === 'pingpong' && $pauseBetween > 0.0001) {
    $t = $fReturnEnd;
    $addPoint($times, $vals, $t, (string) $xNorm);
    $addPoint($times, $vals, min(1.0, $t + $eps), (string) $xBase);
    $addPoint($times, $vals, 1.0, (string) $xBase);
  } else {
    // If looping, reset at the very end to prep next iteration (masked by opacity=0 at end)
    if ($isLooping) {
      $addPoint($times, $vals, max(0.0, 1.0 - $eps), (string) $xNorm);
      $addPoint($times, $vals, 1.0, (string) $xBase);
    } else {
      $addPoint($times, $vals, 1.0, (string) $xNorm);
    }
  }

  $xAnim = [
    'values'   => implode(';', $vals),
    'keyTimes' => implode(';', $times),
  ];
}

// Motion keyPoints/keyTimes
$motionKeyPoints = '';
$motionKeyTimes  = '';
if ($playback === 'pingpong') {

  if ($pauseBetween > 0.0001) {
    $kp = ($direction === 'reverse') ? '1;0;1;1' : '0;1;0;0';
    $kt = '0;' . $fmt($fForwardEnd) . ';' . $fmt($fReturnEnd) . ';1';
    $motionKeyPoints = $kp;
    $motionKeyTimes  = $kt;
  } else {
    $kp = ($direction === 'reverse') ? '1;0;1' : '0;1;0';
    $kt = '0;0.5;1';
    $motionKeyPoints = $kp;
    $motionKeyTimes  = $kt;
  }

} else {
  // loop
  if ($pauseBetween > 0.0001) {
    $t = $fmt($dur / $totalDur);
    $kp = ($direction === 'reverse') ? '1;0;0' : '0;1;1';
    $kt = '0;' . $t . ';1';
    $motionKeyPoints = $kp;
    $motionKeyTimes  = $kt;
  } else {
    $kp = ($direction === 'reverse') ? '1;0' : '0;1';
    $kt = '0;1';
    $motionKeyPoints = $kp;
    $motionKeyTimes  = $kt;
  }
}

// Outer wrapper (builder id/class/attrs + spacing)
$outer = $this->el('div', [
  'class' => [
    'hpb-image-follow-path',
    $uid . '-outer',
    $props['class'] ?? null,
  ],
  'id' => trim((string) ($props['id'] ?? '')) ?: null,
  'style' => sprintf(
    '--ifp-mt:%spx;--ifp-mb:%spx;--ifp-mt-s:%spx;--ifp-mb-s:%spx;',
    htmlspecialchars((string) $mt, ENT_QUOTES, 'UTF-8'),
    htmlspecialchars((string) $mb, ENT_QUOTES, 'UTF-8'),
    htmlspecialchars((string) $mtS, ENT_QUOTES, 'UTF-8'),
    htmlspecialchars((string) $mbS, ENT_QUOTES, 'UTF-8')
  ),
]);

$wrap = $this->el('div', [
  'class' => [$uid],
  'data-hpb-ifp' => $uid,
]);

?>
<?= $outer($props, $attrs) ?>
  <?= $wrap() ?>

<?php if (!empty($props['css'])) : ?>
<style>
<?= $props['css'] . "\n" ?>
</style>
<?php endif; ?>

<style>
.<?= $this->escape($uid . '-outer') ?>::before,
.<?= $this->escape($uid . '-outer') ?>::after{
  content: '';
  display: block;
  height: 0;
}
.<?= $this->escape($uid . '-outer') ?>::before{
  height: var(--ifp-mt, 0px);
}
.<?= $this->escape($uid . '-outer') ?>::after{
  height: var(--ifp-mb, 0px);
}
@media (max-width: 640px){
  .<?= $this->escape($uid . '-outer') ?>::before{
    height: var(--ifp-mt-s, var(--ifp-mt, 0px));
  }
  .<?= $this->escape($uid . '-outer') ?>::after{
    height: var(--ifp-mb-s, var(--ifp-mb, 0px));
  }
}

.<?= $this->escape($uid) ?>{
  position: relative;
  width: 100%;
  height: <?= (int) $height ?>px;
}
.<?= $this->escape($uid) ?> svg{
  width: 100%;
  height: 100%;
  display: block;
  overflow: visible;
}
</style>

<!-- preload to reduce "first frame" flicker -->
<img src="<?= htmlspecialchars($src, ENT_QUOTES, 'UTF-8') ?>"
     alt=""
     width="1" height="1"
     loading="eager"
     decoding="async"
     style="position:absolute;left:-9999px;top:-9999px;width:1px;height:1px;opacity:0;pointer-events:none;"
     aria-hidden="true">

<svg id="<?= htmlspecialchars($uid . '-svg', ENT_QUOTES, 'UTF-8') ?>"
     class="hpb-ifp-svg"
     xmlns="http://www.w3.org/2000/svg"
     viewBox="<?= htmlspecialchars($viewbox, ENT_QUOTES, 'UTF-8') ?>"
     preserveAspectRatio="none"
     aria-hidden="true"
     focusable="false">

  <defs>
    <path id="<?= htmlspecialchars($uid . '-path', ENT_QUOTES, 'UTF-8') ?>"
          d="<?= htmlspecialchars($motionD, ENT_QUOTES, 'UTF-8') ?>"></path>
  </defs>

  <?php if ($showPath) : ?>
    <path d="<?= htmlspecialchars($dFull, ENT_QUOTES, 'UTF-8') ?>"
          fill="<?= htmlspecialchars($fill, ENT_QUOTES, 'UTF-8') ?>"
          stroke="<?= htmlspecialchars($stroke, ENT_QUOTES, 'UTF-8') ?>"
          stroke-width="<?= htmlspecialchars((string) $strokeW, ENT_QUOTES, 'UTF-8') ?>"
          opacity="<?= htmlspecialchars((string) $opacityP, ENT_QUOTES, 'UTF-8') ?>"
          vector-effect="non-scaling-stroke"></path>
  <?php endif; ?>

  <image id="<?= htmlspecialchars($uid . '-img', ENT_QUOTES, 'UTF-8') ?>"
         href="<?= htmlspecialchars($src, ENT_QUOTES, 'UTF-8') ?>"
         xlink:href="<?= htmlspecialchars($src, ENT_QUOTES, 'UTF-8') ?>"
         xmlns:xlink="http://www.w3.org/1999/xlink"
         x="<?= htmlspecialchars((string) $xBase, ENT_QUOTES, 'UTF-8') ?>"
         y="<?= htmlspecialchars((string) $y, ENT_QUOTES, 'UTF-8') ?>"
         width="<?= htmlspecialchars((string) $imgW, ENT_QUOTES, 'UTF-8') ?>"
         height="<?= htmlspecialchars((string) $imgH, ENT_QUOTES, 'UTF-8') ?>"
         opacity="<?= $needsOpacity ? '0' : '1' ?>"
         preserveAspectRatio="xMidYMid slice">

    <?php if ($xAnim) : ?>
      <animate attributeName="x"
               attributeType="XML"
               dur="<?= htmlspecialchars((string) $totalDur, ENT_QUOTES, 'UTF-8') ?>s"
               begin="<?= htmlspecialchars((string) $delay, ENT_QUOTES, 'UTF-8') ?>s"
               repeatCount="<?= htmlspecialchars($repeat, ENT_QUOTES, 'UTF-8') ?>"
               fill="freeze"
               calcMode="linear"
               values="<?= htmlspecialchars($xAnim['values'], ENT_QUOTES, 'UTF-8') ?>"
               keyTimes="<?= htmlspecialchars($xAnim['keyTimes'], ENT_QUOTES, 'UTF-8') ?>" />
    <?php endif; ?>

    <?php if ($opacityAnim) : ?>
      <animate attributeName="opacity"
               attributeType="XML"
               dur="<?= htmlspecialchars((string) $totalDur, ENT_QUOTES, 'UTF-8') ?>s"
               begin="<?= htmlspecialchars((string) $delay, ENT_QUOTES, 'UTF-8') ?>s"
               repeatCount="<?= htmlspecialchars($repeat, ENT_QUOTES, 'UTF-8') ?>"
               fill="freeze"
               calcMode="linear"
               values="<?= htmlspecialchars($opacityAnim['values'], ENT_QUOTES, 'UTF-8') ?>"
               keyTimes="<?= htmlspecialchars($opacityAnim['keyTimes'], ENT_QUOTES, 'UTF-8') ?>" />
    <?php endif; ?>

    <animateMotion dur="<?= htmlspecialchars((string) $totalDur, ENT_QUOTES, 'UTF-8') ?>s"
                   begin="<?= htmlspecialchars((string) $delay, ENT_QUOTES, 'UTF-8') ?>s"
                   repeatCount="<?= htmlspecialchars($repeat, ENT_QUOTES, 'UTF-8') ?>"
                   fill="freeze"
                   rotate="<?= htmlspecialchars($rotate, ENT_QUOTES, 'UTF-8') ?>"
                   calcMode="linear"
                   keyPoints="<?= htmlspecialchars($motionKeyPoints, ENT_QUOTES, 'UTF-8') ?>"
                   keyTimes="<?= htmlspecialchars($motionKeyTimes, ENT_QUOTES, 'UTF-8') ?>">
      <mpath href="#<?= htmlspecialchars($uid . '-path', ENT_QUOTES, 'UTF-8') ?>"
             xlink:href="#<?= htmlspecialchars($uid . '-path', ENT_QUOTES, 'UTF-8') ?>"
             xmlns:xlink="http://www.w3.org/1999/xlink"></mpath>
    </animateMotion>

  </image>
</svg>

<script>
(function(){
  const uid = <?= json_encode($uid) ?>;
  const root = document.querySelector('[data-hpb-ifp="'+uid+'"]');
  if (!root) return;

  const svg = root.querySelector('#'+uid+'-svg');
  if (!svg) return;

  // Pause on hover
  const pauseHover = <?= json_encode($pauseHover) ?>;
  if (pauseHover) {
    root.addEventListener('pointerenter', () => { try { svg.pauseAnimations(); } catch(e) {} }, {passive:true});
    root.addEventListener('pointerleave', () => { try { svg.unpauseAnimations(); } catch(e) {} }, {passive:true});
  }

  // Pause on tap / click (mobile)
  const pauseTap = <?= json_encode($pauseTap) ?>;
  if (pauseTap) {
    let paused = false;
    root.addEventListener('click', () => {
      paused = !paused;
      try {
        if (paused) svg.pauseAnimations();
        else svg.unpauseAnimations();
      } catch(e) {}
    }, {passive:true});
  }

})();
</script>

  <?= $wrap->end() ?>
<?= $outer->end() ?>